<?php

namespace App\Http\Controllers;

use App\Models\Classe;
use App\Models\Sanction;
use App\Models\LevelMatter;
use App\Models\MoyenneDecoupageMatiere;
use App\Models\CuttingSchoolYear;
use App\Models\ApprovedMoyenne;
use App\Jobs\SaveConduiteJob;
use App\Jobs\CalculMoyenneTrimestielleJob;
use App\Jobs\CalculMoyenneBilanJob;
use App\Jobs\CalculResultatClasseJob;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;

class ConduiteController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index($id)
    {
        try{
            $type = ['secondaire' => 'general', 'technique' => 'technique', 'superieur' => 'superieur'];
            $class = Classe::find($id);
            $cutting = CuttingSchoolYear::where('type', $type[$class['type']])->where('school_year_id', $class['school_year_id'])->get();
            return view('pages.conduite.index',[
                'classe' => $class,
                'data' => $this->getDataSanction($cutting, $class),
                'cndte' => $this->getMattertConduite($class['level_id'])
            ]);
        }
        catch (\Exception $e) {
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue.'
            ]);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try{
            $val = $request->validate([
                'classe' => 'required|string',
                'cutting' => 'required|string',
                'matter' => 'required|string'
            ]);
            $class = Classe::find($val['classe']);
            ApprovedMoyenne::create([
                'classe_id' => $val['classe'],
                'level_matter_id' => $val['matter'],
                'cutting_school_year_id' => $val['cutting'],
                'approved' => '1',
            ]);

            // Utilisation de jobs .............................................
            Bus::chain([
                new SaveConduiteJob($this->getSanctions($class, $val['cutting']), $class, $val['cutting'], $val['matter']),
                new CalculMoyenneTrimestielleJob($this->getStudent($class), $val['classe'], $val['cutting'], auth()->user()->school_id),
                new CalculMoyenneBilanJob($this->getStudent($class), $val['classe'], $val['cutting'], $this->bilanMatter($val['matter']), auth()->user()->school_id),
                new CalculResultatClasseJob($val['classe'], auth()->user()->school_id, $val['cutting'])
            ])->dispatch();
            
            return back()->with([
                'str' => 'info',
                'msg' => 'Les moyenne de conduite sera connue dans quelque temps !'
            ]);
        }
        catch (\Exception $e) {
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue.'
            ]);
        }
    }

    /**
     * Display the specified resource.
     */
    public function search(Request $request)
    {
        try{
            $val = $request->validate(['id' => 'required|integer']);
            $data = CuttingSchoolYear::find($val['id']);
            return ucfirst($data->cutting['libelle']);
        }
        catch (\Exception $e) {
            return back()->with([
                'str' => 'danger',
                'msg' => 'Une erreur est survenue.'
            ]);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    
    protected function getDataSanction($cutting, $class)
    {
        $datas = []; $i = 1;
        $matter = $this->getMattertConduite($class['level_id']);
        $data = [1 => 'realtime-reorder', 2 => 'basic-col-reorder', 3 => 'saving-reorder', 4 => 'predefine-reorder', 5 => null];
        foreach($cutting as $item){
            $datas[] = [
                'id' => $item->id,
                'tab' => $data[$i],
                'libelle' => $item->cutting['libelle'],
                'actif' => $item->status,
                'approved' => $this->approvedMoyenneMatter($class->id, $matter->id, $item->id),
                'data' => $this->getSanctions($class, $item->id)
            ];
            $i++;
        }
        return $datas;
    }


    protected function getSanctions($class, $cutting)
    {
        $students = $this->getStudent($class);
        $matter = $this->getMattertConduite($class['level_id']);
        $studts = [];
        foreach($students as $item){
            $verify = Sanction::where('inscription_id', $item->id)->where('cutting_school_year_id', $cutting)->where('discipline', '1')->first();
            $rang = MoyenneDecoupageMatiere::where('inscription_id', $item->id)->where('level_matter_id', $matter->id)->where('cutting_school_year_id', $cutting)->first();
            $studts[] = [
                'id' => $item->id,
                'name' => strtoupper($item->first_name).' '.ucwords($item->last_name),
                'sexe' => strtoupper($item->sexe),
                'matricule' => $item->matricule,
                'photo' => $item->photo,
                'assiduite' => $this->cmpteNbreSanction($item->id, $cutting, 1), // id = 1
                'discipline' => $this->cmpteNbreSanction($item->id, $cutting, 2), // id = 2
                'moralite' => $this->cmpteNbreSanction($item->id, $cutting, 3), // id = 3
                'tenue' => $this->cmpteNbreSanction($item->id, $cutting, 4), // id = 4
                'moyen' => $rang ? $rang->moyenne:$this->calculSanctionPrivisoire($item->id, $cutting),
                'rang' => $rang ? $rang->rang:'---',
                'conseil' => $verify ? 1:0
            ];
        }
        return $studts;
    }


    protected function calculSanctionPrivisoire($student, $cutting)
    {
        $verify = Sanction::where('inscription_id', $student)->where('cutting_school_year_id', $cutting)->where('discipline', '1')->first();
    
        $total = 0;
        if($verify){ // moyenne par defaut est 05 dans ce cas
            $total = 5; 
        }
        else{ // moyenne par defaut est 20
            $assiduite = $this->cmpteNbreSanction($student, $cutting, 1);
            $discipline = $this->cmpteNbreSanction($student, $cutting, 2);
            $moralite = $this->cmpteNbreSanction($student, $cutting, 3);
            $tenue = $this->cmpteNbreSanction($student, $cutting, 4);
            $val = $assiduite > 10 ? 6:($assiduite*0.5);
            $total = (20 - ($val+$discipline+$moralite+($tenue*0.5)));
        }
        $moy = $total ? number_format(($total), 2, '.', ' '):'nc';
        return $total ? ($moy < 10 ? '0'.$moy:$moy):$moy;
    }


    protected function cmpteNbreSanction($student, $cutting, $typeSanct)
    {
        $count = Sanction::where('inscription_id', $student)->where('motif_sanction_id', $typeSanct)->where('cutting_school_year_id', $cutting)->count();
        return $count;
    }


    protected function getMattertConduite($level)
    {
        $cndte = LevelMatter::where('school_id', auth()->user()->school_id)->where('level_id', $level)->where('matter_id', '15')->first();
        return $cndte;
    }


    protected function approvedMoyenneMatter($class, $matter, $decoupage)
    {
        $first = ApprovedMoyenne::where('cutting_school_year_id', $decoupage)->where('classe_id', $class)->where('level_matter_id', $matter)->first();
        return $first->approved ?? null;
    }


    protected function getStudent($class)
    {
        $dts = DB::table('students')
        ->join('inscriptions', 'students.id', '=', 'inscriptions.student_id')
        ->select('inscriptions.id', 'students.first_name', 'students.last_name', 'students.matricule', 'students.sexe', 'students.photo', 'inscriptions.redoublant')
        ->where('inscriptions.classe_id', '=', $class->id)
        ->where('inscriptions.school_year_id', '=', $class->school_year_id)
        ->orderBy('students.first_name')->orderBy('students.last_name')
        ->get();

        return $dts;
    }


    protected function bilanMatter($valeur)
    {
        $val = DB::table('matters')
        ->join('level_matter', 'matters.id', '=', 'level_matter.matter_id')
        ->join('libelle_bilan_matter', 'matters.id', '=', 'libelle_bilan_matter.matter_id')
        ->select('libelle_bilan_matter.libelle_bilan_id')->where('level_matter.id', '=', $valeur)->first();
        return $val->libelle_bilan_id;
    }
}